#include "sharedtask.h"

/**
 * SharedTask Explanation
 *
 * PURPOSE:
 *   To reduce RAM requirement of firmware components by combining
 * tasks into a single OS task. -- It is similar to the OsTimer framework,
 * but allows for finer control over task execution context and timing.
 *
 * REQUIREMENTS:
 *   In order to be successfully combined, each task must be of the form
 * of a state-machine with known sleep lengths between state transitions.
 * For example, a task which consists of a while(true) loop, a number of
 * if-statements, and a HwSleepMs(x) call at the end of each control path
 * is a good candidate for being combined.
 *   The tasks in question must be refactored as functions. The functions
 * take only a 32-bit param (*) and return an integer giving the amount of 
 * time that they wish to sleep for, in milliseconds.
 *
 * IMPLEMENTATION:
 *   The caller sets up a config table of function pointers to the subtasks.
 * This config is passed into the SharedTask constructor. The SharedTask
 * keeps track of the time-to-sleep for each subtask. Upon each return, the
 * SharedTask updates the amount of time the subtask wishes to sleep and then
 * runs the next ready subtask. If no subtasks are ready, the whole task sleeps
 * for the minimum of the time-to-sleeps of all subtasks.
 *   
 * (*) For calling C++ methods as subtasks, the SharedTask supports a 32-bit 
 * parameter which should be a pointer to the class instance.
 */

#define SHAREDTASK_MAX_SLEEPTIME    120000
#define SHAREDTASK_MIN_SLEEPTIME    1

/**
 * \author    Chris Merck
 * \brief     Share a single task between several subtasks
 * \date      21 JUNE 2013
 * \param     pTaskList = list of iNumTasks function pointers.
 *                        each func ptr takes no arguments and returns
 *                        an INT32 containing the amount of time it wants
 *                        to sleep in milliseconds. A return value of 0 will
 *                        yield to let other tasks run. A negative value will
 *                        cause task to be never run again.
 * \param     iNumTasks
**/
SharedTask::SharedTask(SHARED_SUBTASK_ENTRY *pTaskList, UINT8 iNumTasks, UINT32 param)
{
    UINT8 i;

    m_iNumTasks = iNumTasks;

    m_param = param;

    m_pTasks = (SharedTask_TaskInfo *) OsAllocMemory(iNumTasks * sizeof(SharedTask_TaskInfo));
    if (!m_pTasks)
        return;

    for (i = 0; i < iNumTasks; i++)
    {
        m_pTasks[i].iSleepTime = 0; // ready to run
        m_pTasks[i].pEntryPoint = *(pTaskList+i);
    }
}

/**
 * \author    Chris Merck
 * \brief     Entry point for shared task.
 * \note      Runs until destructor is called or all tasks finish.
 * \date      21 JUNE 2013
**/
void SharedTask::Run()
{
    UINT8 i; // current task index
    INT32 lastSleepTime = 0;
    INT32 nextSleepTime;
    UINT8 tasksLeft = m_iNumTasks;

    if (!m_pTasks)
        return;

    while (true && m_pTasks)
    {
        nextSleepTime = SHAREDTASK_MAX_SLEEPTIME;

        // round-robin through tasks,
        //  running any ready tasks,
        //  and service timers.
        for (i = 0; i<m_iNumTasks; i++)
        {
            // if task is valid
            if (m_pTasks[i].pEntryPoint.SubTaskPtr)
            {
                // account for previous sleep
                m_pTasks[i].iSleepTime -= lastSleepTime;

                // if task is ready to run
                if (m_pTasks[i].iSleepTime <= 0)
                {
                    // run task now and schedule next run
                    m_pTasks[i].iSleepTime = (m_pTasks[i].pEntryPoint.SubTaskPtr)(m_pTasks[i].pEntryPoint.argList);

                    // check if task does not want to be called again
                    if (m_pTasks[i].iSleepTime < 0)
                    {
                        m_pTasks[i].pEntryPoint.SubTaskPtr = 0;
                        tasksLeft--;
                    }
                }
            }

            // if task requesting valid sleeptime (0 means yield but rerun as soon as possible)
            if (m_pTasks[i].iSleepTime >= 0)
            {
                // keep track of shortest interval until next scheduled task
                nextSleepTime = min(nextSleepTime, m_pTasks[i].iSleepTime);
                if(nextSleepTime == 0)
                	nextSleepTime = SHAREDTASK_MIN_SLEEPTIME;
            }
        }

        // quit if no tasks still running
        if (!tasksLeft)
        {
            break;
        }

        // sleep until next task needs to run
        if (nextSleepTime)
        {
            HwDelayMsec(nextSleepTime);
        }

        // save sleep time
        lastSleepTime = nextSleepTime;
    }
}

/**
 * \author    Chris Merck
 * \brief     Shutdown shared task.
 * \date      21 JUNE 2013
 * \note      The task will run in background until current subtask finishes
**/
SharedTask::~SharedTask()
{
    (void)OsFreeMemory((void*)m_pTasks);
    m_pTasks = 0;
    
}
